package com.aubay.formations.nr.utils;

import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import com.aubay.formations.nr.entities.Country;
import com.aubay.formations.nr.entities.Employee;
import com.aubay.formations.nr.entities.Usage;

/**
 * Dummy Data Generator
 *
 * @author jbureau@aubay.com
 * @formatter:off
 */
public class DataGenerator {

	private static final SecureRandom random = new SecureRandom();

	private static final List<String> COUNTRIES  = Arrays.asList("Afghanistan", "Afrique du Sud", "Albanie", "Algérie","Allemagne", "Andorre", "Angola", "Anguilla", "Antigua-et-Barbuda", "Antilles Néerlandaises","Arabie Saoudite", "Argentine", "Arménie", "Aruba", "Australie", "Autriche", "Azerbaïdjan", "Bahamas","Bahreïn", "Bangladesh", "Barbade", "Belgique", "Belize", "Bénin", "Bermudes", "Bhoutan", "Biélorussie","Birmanie (Myanmar)", "Bolivie", "Bosnie-Herzégovine", "Botswana", "Brésil", "Brunei", "Bulgarie","Burkina Faso", "Burundi", "Cambodge", "Cameroun", "Canada", "Cap-vert", "Chili", "Chine", "Chypre","Colombie", "Comores", "Corée du Nord", "Corée du Sud", "Costa Rica", "Côte d'Ivoire", "Croatie", "Cuba","Danemark", "Djibouti", "Dominique", "Égypte", "Émirats Arabes Unis", "Équateur", "Érythrée", "Espagne","Estonie", "États Fédérés de Micronésie", "États-Unis", "Éthiopie", "Fidji", "Finlande", "France", "Gabon","Gambie", "Géorgie", "Géorgie du Sud et les Îles Sandwich du Sud", "Ghana", "Gibraltar", "Grèce", "Grenade","Groenland", "Guadeloupe", "Guam", "Guatemala", "Guinée", "Guinée-Bissau", "Guinée Équatoriale", "Guyana","Guyane Française", "Haïti", "Honduras", "Hong-Kong", "Hongrie", "Île Christmas", "Île de Man","Île Norfolk", "Îles Åland", "Îles Caïmanes", "Îles Cocos (Keeling)", "Îles Cook", "Îles Féroé","Îles Malouines", "Îles Mariannes du Nord", "Îles Marshall", "Îles Pitcairn", "Îles Salomon","Îles Turks et Caïques", "Îles Vierges Britanniques", "Îles Vierges des États-Unis", "Inde", "Indonésie","Iran", "Iraq", "Irlande", "Islande", "Israël", "Italie", "Jamaïque", "Japon", "Jordanie", "Kazakhstan","Kenya", "Kirghizistan", "Kiribati", "Koweït", "Laos", "Le Vatican", "Lesotho", "Lettonie", "Liban","Libéria", "Libye", "Liechtenstein", "Lituanie", "Luxembourg", "Macao", "Madagascar", "Malaisie", "Malawi","Maldives", "Mali", "Malte", "Maroc", "Martinique", "Maurice", "Mauritanie", "Mayotte", "Mexique","Moldavie", "Monaco", "Mongolie", "Monténégro", "Montserrat", "Mozambique", "Namibie", "Nauru", "Népal","Nicaragua", "Niger", "Nigéria", "Niué", "Norvège", "Nouvelle-Calédonie", "Nouvelle-Zélande", "Oman","Ouganda", "Ouzbékistan", "Pakistan", "Palaos", "Panama", "Papouasie-Nouvelle-Guinée", "Paraguay","Pays-Bas", "Pérou", "Philippines", "Pologne", "Polynésie Française", "Porto Rico", "Portugal", "Qatar","République Centrafricaine", "République de Macédoine", "République Démocratique du Congo","République Dominicaine", "République du Congo", "République Tchèque", "Réunion", "Roumanie", "Royaume-Uni","Russie", "Rwanda", "Saint-Kitts-et-Nevis", "Saint-Marin", "Saint-Pierre-et-Miquelon","Saint-Vincent-et-les Grenadines", "Sainte-Hélène", "Sainte-Lucie", "Salvador", "Samoa","Samoa Américaines", "Sao Tomé-et-Principe", "Sénégal", "Serbie", "Seychelles", "Sierra Leone", "Singapour","Slovaquie", "Slovénie", "Somalie", "Soudan", "Sri Lanka", "Suède", "Suisse", "Suriname","Svalbard et Jan Mayen", "Swaziland", "Syrie", "Tadjikistan", "Taïwan", "Tanzanie", "Tchad","Terres Australes Françaises", "Thaïlande", "Timor Oriental", "Togo", "Tonga", "Trinité-et-Tobago","Tunisie", "Turkménistan", "Turquie", "Tuvalu", "Ukraine", "Uruguay", "Vanuatu", "Venezuela", "Viet Nam","Wallis et Futuna", "Yémen", "Zambie", "Zimbabwe");
	private static final List<String> FIRSTNAMES = Arrays.asList("Axel", "Arthur", "Evan", "Adrien", "Alexis","Antoine", "Adam", "Alexandre", "Baptiste", "Enzo", "Julien", "Leo", "Lucas", "Matteo", "Mael", "Maxime","Gabriel", "Raphael", "Pierre", "Quentin", "Hugo", "Romain", "Theo", "Tom", "Jules", "Nicolas", "Louis","Mathis", "Nathan", "Paul", "Valentin", "Ethan", "Kylian", "Matheo", "Aaron", "Antonin", "Anthony", "Ayoub","Bastien", "Alan", "Aymeric", "Bryan", "Charles", "Elias", "Dorian", "Dylan", "Alex", "Augustin", "Alban","Aurelien", "Benjamin", "Arnaud", "Gael", "Gabin", "Guillaume", "Florian", "Corentin", "Jean", "Jeremy","Diego", "Emilien", "Esteban", "David", "Etienne", "Damien", "Erwan", "Lukas", "Loic", "Lorenzo", "Mathias","Matthieu", "Gaetan", "Gaspard", "Morgan", "Rafael", "Kilian", "Samuel", "Simon", "Kevin", "Sacha","Tristan", "Victor", "Jordan", "Killian", "Marius", "Vincent", "Martin", "Yann", "Mateo", "William", "Luca","Remi", "Oscar", "Robin", "Noé", "Tony", "Tiago", "Thibaut", "Bilal", "Thibault", "Eliot", "Elouan", "Ilan","Eliott", "Julian", "Kyllian", "Ewan", "Luka");
	private static final List<String> LASTNAMES  = Arrays.asList("MARTIN", "BERNARD", "PETIT", "THOMAS", "DUBOIS","RICHARD", "MICHEL", "MOREAU", "ROBERT", "DURAND", "ROUX", "SIMON", "LAURENT", "LAMBERT", "LEROY", "GIRARD","LEFEBVRE", "BERTRAND", "MOREL", "DAVID", "ROUSSEAU", "FOURNIER", "MERCIER", "GARNIER", "HENRY", "DUPONT","BONNET", "ROY", "VINCENT", "GUERIN", "BLANC", "LEFEVRE", "PERRIN", "ANDRE", "MARCHAND", "MATHIEU","MASSON", "FRANCOIS", "LEGRAND", "CHEVALIER", "CLEMENT", "GERARD", "ROBIN", "FONTAINE", "NICOLAS","GAUTIER", "BARBIER", "MORIN", "NOEL", "DENIS", "BRUN", "GIRAUD", "ROUSSEL", "COLIN", "LEMAIRE", "FAURE","RENARD", "DUMONT", "AUBERT", "DUVAL", "DUFOUR", "JOLY", "BOURGEOIS", "GAILLARD", "BLANCHARD", "GAUTHIER","LECLERC", "BRUNET", "MEUNIER", "MEYER", "ROGER", "BOYER", "PICARD", "BRETON", "MULLER", "PIERRE", "BAILLY","BOUCHER", "LUCAS", "ARNAUD", "CHARPENTIER", "FABRE", "LEROUX", "ROCHE", "GUILLAUME", "ADAM", "OLIVIER","MARCHAL", "HUBERT", "AUBRY", "PHILIPPE", "LEMOINE", "BERGER", "CARON", "DESCHAMPS", "RENAUD", "JACQUET","FLEURY", "LECOMTE", "HUET", "PREVOST", "BENOIT", "PELLETIER", "GILLET", "CARRE", "ROYER", "ROLLAND","DUPUIS", "RIVIERE", "LACROIX", "GUILLOT", "PARIS", "DUMAS", "REY", "REMY", "JEAN", "GUYOT", "MOULIN","LANGLOIS", "LECLERCQ", "CARPENTIER", "LOUIS", "MAILLARD", "MILLET", "BARON", "MARIE", "COLLIN", "COLLET","VIDAL", "POIRIER", "HUMBERT", "GERMAIN", "ANTOINE", "JACOB", "BESSON", "MONNIER", "BOUVIER", "PERROT","JULIEN", "COUSIN", "MARECHAL", "RENAULT", "LEBRUN", "HERVE", "BARRE", "LEGER", "BOULANGER", "CORDIER","CARLIER", "GUICHARD", "CLERC", "BUISSON", "CAMUS", "DIDIER", "CHARLES", "BIGOT", "FRANCE", "DANIEL","MALLET", "BARTHELEMY");
	private static final List<String> URIS  = Arrays.asList("DELETE /stats", "POST /employees", "GET /employees", "GET /employees/top", "GET /countries", "GET /stats");
	private static final List<Float> URIS_DEFAULT_DURATION  = Arrays.asList(20000F, 1000F, 25000F, 20000F, 2000F, 70000F);
	private static final List<Float> URIS_DEFAULT_QUERIES  = Arrays.asList(1F, 10F, 3000F, 2500F, 1F, 1F);
	private static final List<Float> URIS_DEFAULT_WEIGHT  = Arrays.asList(0F, 800F, 3000000F, 2000000F, 5000F, 5000000F);
	private static final List<Integer> URIS_CALLS  = Arrays.asList(1, 379, 3543, 9365, 13576, 823);

	private DataGenerator() {
		// Hide constructor
	}

	/**
	 * Generate countries list
	 * @return List<Country>
	 */
	public static List<Country> generateCountries() {
		return COUNTRIES.stream().map(p -> Country.builder().code(p.substring(0, 2)).labelFr(p).labelEn(p).build()).collect(Collectors.toList());
	}

	/**
	 * Get random amount between min and max
	 * @param min
	 * @param max
	 * @return Integer
	 */
	private static Integer getRandomAmount(final int min, final int max) {
		return (int) (min + random.nextInt(max - min));
	}

	/**
	 * Get random double between min and max
	 * @param min
	 * @param max
	 * @return Integer
	 */
	private static Double getRandomDouble(final double min, final double max) {
		return (double) (min + random.nextDouble(max - min));
	}

	/**
	 * Get random at given percentage
	 * @return true at percent%
	 */
	private static boolean getRandomBooleanAtPercentage(final int percent) {
		return random.nextInt(100) < percent;
	}

	/**
	 * Get random date since this number of days
	 * @return Date
	 */
	private static Date getRandomDateSince(final int maxDaysBefore) {
		final var cal = Calendar.getInstance();
		final var nbDaySinceThisDate = random.nextInt(maxDaysBefore);
		cal.add(Calendar.DAY_OF_MONTH, -nbDaySinceThisDate);
		return cal.getTime();
	}

	/**
	 * Get date X days before today
	 * @return Date
	 */
	private static Date getDateFromToday(final int nbDays) {
		final var cal = Calendar.getInstance();
		cal.add(Calendar.DAY_OF_MONTH, -nbDays);
		return cal.getTime();
	}

	/**
	 * Get random element in list
	 * @param elements
	 * @return one element of this list
	 */
	private static String getRandomElement(final List<String> elements) {
		return elements.get(random.nextInt(elements.size()));
	}

	/**
	 * Generate employees list
	 * @param maxInTeam
	 * @param maxDeep
	 * @return List<Employee>
	 */
	public static List<Employee> generateEmployees(final int maxInTeam, final int maxDeep) {
		return generateEmployees(maxInTeam, maxDeep, null);
	}

	/**
	 * Generate employees list
	 * @param empByTeam
	 * @param maxDeep
	 * @param manager
	 * @return List<Employee>
	 */
	private static List<Employee> generateEmployees(final int empByTeam, final int maxDeep, final Employee manager) {
		final List<Employee> result = new ArrayList<>();
		final var countryLb = getRandomBooleanAtPercentage(10) || manager == null ? getRandomElement(COUNTRIES) : manager.getCountry().getLabelFr();
		final var firstname = getRandomElement(FIRSTNAMES);
		final var lastname = getRandomElement(LASTNAMES);
		final var country = Country.builder().code(countryLb.substring(0, 2)).labelFr(countryLb).labelEn(countryLb).build();
		final var entryDate = getRandomDateSince(20 * 365);
		final var resigned = getRandomBooleanAtPercentage(10) && maxDeep > 2;
		final var salary = getRandomAmount(20000, 60000);
		final var emp = Employee.builder()
				.firstname(firstname)
				.lastname(lastname)
				.entryDate(entryDate)
				.resigned(resigned)
				.salary(salary)
				.country(country)
				.manager(manager)
				.build();
		result.add(emp);
		if (maxDeep > 1) {
			for (var i = 0; i < getRandomAmount(1,empByTeam); i++) {
				result.addAll(generateEmployees(empByTeam, maxDeep - 1, emp));
			}
		}
		return result;
	}

	/**
	 * Generate nb random usages
	 * @param nb
	 * @return
	 */
	public static List<Usage> generateUsages(final int nb) {
		final List<Usage> usages = new ArrayList<>();
		for(var j = 1 ; j < URIS.size() ; j++) {
			final var uri = URIS.get(j);
			var pente = 0F;
			var duration = URIS_DEFAULT_DURATION.get(j);
			var queries = URIS_DEFAULT_QUERIES.get(j);
			var weight = URIS_DEFAULT_WEIGHT.get(j);
			final var nbUsages = URIS_CALLS.get(j);
			for(var i = 0 ; i < nbUsages ; i++) {
				if(i % (nbUsages / 10) == 0) { // 10 fois
					pente += getRandomDouble(-100, 100); // variations "naturelles"
					if(getRandomAmount(0,10) <= 2) { // Sauts aléatoires (corrections ou dégradation prononcées)
						duration = (float) (duration * getRandomDouble(0.1, 1.9));
						queries = (float) (queries * getRandomDouble(0.1, 1.9));
						weight = (float) (weight * getRandomDouble(0.1, 1.9));
					}
				}
				duration += pente * duration / 100 / nbUsages * getRandomDouble(0.50, 1.5).floatValue();
				queries += pente * queries / 100 / nbUsages * getRandomDouble(0.50, 1.5).floatValue();
				weight += pente * weight / 100 / nbUsages * getRandomDouble(0.50, 1.5).floatValue();
				usages.add(Usage.builder()
						.date(getDateFromToday(360 - 360 * i / nbUsages))
						.duration(duration > 50 ? duration.intValue() : 50)
						.queries(queries >= 1 ? queries.intValue() : 1)
						.weight(weight >= 500 ? weight.intValue() : 500)
						.uri(uri)
						.build());
			}
		}
		return usages;
	}
}
