package com.aubay.formations.nr.config;

import java.util.List;

import javax.sql.DataSource;
import org.springframework.transaction.annotation.Transactional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpStatus;
import org.springframework.orm.jpa.JpaTransactionManager;
import org.springframework.orm.jpa.LocalContainerEntityManagerFactoryBean;
import org.springframework.orm.jpa.vendor.HibernateJpaVendorAdapter;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.config.Customizer;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configurers.CorsConfigurer;
import org.springframework.security.config.annotation.web.configurers.CsrfConfigurer;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.provisioning.JdbcUserDetailsManager;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.authentication.HttpStatusEntryPoint;
import org.springframework.security.web.csrf.CookieCsrfTokenRepository;
import org.springframework.security.web.csrf.CsrfTokenRequestAttributeHandler;
import org.springframework.transaction.CannotCreateTransactionException;

import com.aubay.formations.nr.repositories.UserRepository;
import com.aubay.formations.nr.services.EmployeeService;
import com.fasterxml.jackson.databind.ObjectMapper;

import jakarta.persistence.EntityManagerFactory;

/**
 * Web security configuration
 *
 * @formatter:off
 * @author jbureau@aubay.com
 */
@Configuration
@EnableWebSecurity
public class WebSecurityConfig {

	@Autowired
	private ObjectMapper mapper;

	@Value("${frontend.url}")
	private String frontendUrl;

	@Bean
	public UserDetailsService userDetailsService(final DataSource ds) {
		return new JdbcUserDetailsManager(ds) {

			@Autowired
			private UserRepository userRepository;

			@Autowired
			private EmployeeService employeeService;

			@Override
			@Transactional
			public UserDetails loadUserByUsername(final String username) throws UsernameNotFoundException {
				final var user = userRepository.findById(username);
				if(!user.isPresent()) {
					throw new UsernameNotFoundException("L'utilisateur " + username + " n'a pas été trouvé");
				}
				employeeService.filterResignedEmployees(user.get().getEmployee());
				employeeService.hibernateRecursiveInitialization(user.get().getEmployee());
				return user.get();
			}
		};
	} 
	
	@Bean
    public LocalContainerEntityManagerFactoryBean entityManagerFactory(DataSource dataSource) {
        LocalContainerEntityManagerFactoryBean emfb = new LocalContainerEntityManagerFactoryBean();
        emfb.setDataSource(dataSource);
        emfb.setPackagesToScan("com.aubay.formations.nr.entities");
        emfb.setJpaVendorAdapter(new HibernateJpaVendorAdapter());

        // Additional configuration here
        return emfb;
    }

    @Bean
    public JpaTransactionManager transactionManager(EntityManagerFactory entityManagerFactory) {
        JpaTransactionManager transactionManager = new JpaTransactionManager();
        transactionManager.setEntityManagerFactory(entityManagerFactory);
        return transactionManager;
    }
	
	@Bean
	public SecurityFilterChain filterChain(final HttpSecurity http) throws Exception {
		// API Security
		http.authorizeHttpRequests(authorize -> authorize.requestMatchers("/tp/**", "/login").permitAll().requestMatchers("/**").authenticated());
		// Handle access rejections
		http.exceptionHandling(exceptionHandling -> exceptionHandling.authenticationEntryPoint(new HttpStatusEntryPoint(HttpStatus.UNAUTHORIZED)));
		// Enable CORS
		http.cors(corsConfiguration());
		// Enable CSRF by Cookies
		http.csrf(csrfConfiguration());
		// Authentication process
		http.formLogin(form -> form
			.loginPage("/login")
			.successHandler((request, response, authentication) -> {
				response.setStatus(HttpStatus.OK.value());
				mapper.writeValue(response.getOutputStream(), authentication.getPrincipal());
			})
			.failureHandler((request, response, exception) -> {
				response.setStatus(HttpStatus.UNAUTHORIZED.value());
				if(exception.getCause() instanceof CannotCreateTransactionException) {
					response.getOutputStream().write("La base de données est inaccessible".getBytes());
				} else if(exception instanceof BadCredentialsException) {
					response.getOutputStream().write("Les identifiants sont incorrects".getBytes());
				} else {
					response.getOutputStream().write("L'authentification a échouée".getBytes());
				}
			})
		);
		// Logout
		http.logout(logout->logout.logoutSuccessHandler((request, response, authentication) ->
			response.setStatus(HttpStatus.OK.value())
		));
		return http.build();
	}

	private Customizer<CorsConfigurer<HttpSecurity>> corsConfiguration() {
		return cors -> cors.configurationSource(request -> {
            var corsConfiguration = new org.springframework.web.cors.CorsConfiguration();
            corsConfiguration.setAllowedOrigins(List.of(frontendUrl));
            corsConfiguration.setAllowedMethods(List.of("*"));
            corsConfiguration.setAllowedHeaders(List.of("*"));
            corsConfiguration.setAllowCredentials(true);
            return corsConfiguration;
        });
	}

	private Customizer<CsrfConfigurer<HttpSecurity>> csrfConfiguration() {
		return csrf -> {
			CsrfTokenRequestAttributeHandler csrfTokenRequestAttributeHandler = new CsrfTokenRequestAttributeHandler();
			csrfTokenRequestAttributeHandler.setCsrfRequestAttributeName(null);
			csrf.csrfTokenRepository(CookieCsrfTokenRepository.withHttpOnlyFalse())
			    .csrfTokenRequestHandler(csrfTokenRequestAttributeHandler);
		};
	}

}